unit TriangC;
//*******************************************************
// Triangular array class.
//*******************************************************
// Copyright (C) 1998 John Wiley & Sons, Inc.
// All rights reserved. See additional copyright
// information in Readme.txt.
//
// Revised (C) 1999 Andrzej Grayski, HELION Publ.
//
//*******************************************************

interface

uses
    Dialogs, SysUtils, Classes,
    ExtCtrls, Windows, Graphics;

type
    String10 = String[10];
    TTriangularArray = class(TObject)
        private
            LinearVector : array of String10;
            Rows         : Longint;      // The number of rows.
            UseDiagonal  : Boolean;      // True if the array includes the diagonal.
            function AtoB(i, j : Integer) : Integer;

        public
            constructor Create(num_rows : Integer; include_diagonal : Boolean);
            destructor Destroy; override;
            procedure SetValue(r, c : Integer; new_value : String10);
            function GetValue(r, c : Integer) : String10;
    end;

implementation

// Create the proper kind of triangular array.
constructor TTriangularArray.Create(num_rows : Integer; include_diagonal : Boolean);
begin
    if (num_rows < 1) then
        raise EInvalidOperation.Create(
            'Array must have at least one row.');

    // Allocate memory and perform inherited initialization.
    inherited Create;

    UseDiagonal := include_diagonal;

    // Allocate room for the entries.
    Rows := num_rows;

    {
    if (UseDiagonal) then
        GetMem(Items, Round((Rows * (Rows + 1) / 2)) * SizeOf(String10))
    else
        GetMem(Items, Round((Rows * (Rows - 1) / 2)) * SizeOf(String10));
     }
    if (UseDiagonal)
    then
      SetLength(LinearVector, (Rows * (Rows + 1)) div 2)
    else
      SetLength(LinearVector, (Rows * (Rows - 1)) div 2)



end;

// Free the array memory.
destructor TTriangularArray.Destroy;
begin
  LinearVector := NIL;
  inherited Destroy;
end;

// Set an item's value.
procedure TTriangularArray.SetValue(r, c : Integer; new_value : String10);
var
    x : Integer;
begin
  x := AtoB(r, c);
  LinearVector[x] := new_value;
end;

// Get an item's value.
function TTriangularArray.GetValue(r, c : Integer) : String10;
var
    x : Integer;
begin
    x := AtoB(r, c);
    GetValue := LinearVector[x];
end;

// Przeksztacenie indeksw i oraz j macierzy trjktnej A
// na indeks wektora linearyzujcego B.
function TTriangularArray.AtoB(i, j : Integer) : Integer;
var
    tmp : Integer;
begin
    if ((i < 0) or (i >= Rows) or
        (j < 0) or (j >= Rows))
    then
        raise EInvalidOperation.CreateFmt(
            'Indeksy %d oraz %d musz posiada warto z przedziau %d - %d.',
            [i, j, 0, Rows - 1]);

    if ((not UseDiagonal) and (i = j)) then
        raise EInvalidOperation.Create(
            'Przektna tej macierzy nie moe by uwzgldniona.');

    // Zamiana indeksw, aby byo i > j
    if (i < j) then
    begin
        tmp := i;
        i := j;
        j := tmp;
    end;

    if (UseDiagonal)
    then
      i := i + 1;
    AtoB := (i * (i - 1)) div 2 + j;
end;

end.
