unit SerDraw;
//*******************************************************
// The TDrawingCommand class base class from which other
// command classes are derived.
//
// The class defines the abstract virtual Draw,
// Serialization, and Deserialize methods implemented by
// derived serializable drawing command classes.
//
// Deriving classes from this class allows a program to
// treat the commands polymorphically as TDrawingCommand
// objects.
//*******************************************************
// Copyright (C) 1998 John Wiley & Sons, Inc.
// All rights reserved. See additional copyright
// information in Readme.txt.
//*******************************************************

interface

uses SysUtils, Forms, Graphics, ExtCtrls, Serials;

type
    TDrawingCommand = class(TObject)
        protected
            x1, y1, x2, y2 : Integer;
            color          : TColor;
            style          : TBrushStyle;

        public
            procedure SetPosition(new_x1, new_y1, new_x2, new_y2 : Integer);
            procedure SetColorAndStyle(new_color : TColor; new_style : TBrushStyle);
            procedure Draw(cvs : TCanvas; permanent : Boolean); virtual; abstract;
            function Serialization : String; virtual;
            procedure Deserialize(txt : String); virtual;
    end;

implementation

const
    // Constants for serializing and deserializing.
    DEF_x1 = 0;
    DEF_y1 = 0;
    DEF_x2 = 100;
    DEF_y2 = 100;
    DEF_color = clBlack;
    DEF_style = bsSolid;

procedure TDrawingCommand.SetPosition(new_x1, new_y1, new_x2, new_y2 : Integer);
begin
    x1 := new_x1;
    y1 := new_y1;
    x2 := new_x2;
    y2 := new_y2;
end;

procedure TDrawingCommand.SetColorAndStyle(new_color : TColor; new_style : TBrushStyle);
begin
    color := new_color;
    style := new_style;
end;

// Return the basic serialization that stores the
// position color, and style information.
function TDrawingCommand.Serialization : String;
const
    CR = #13#10;
begin
    Result := '';
    if (x1 <> DEF_x1) then Result := Result + Format('  x1(%d)', [x1]) + CR;
    if (y1 <> DEF_y1) then Result := Result + Format('  y1(%d)', [y1]) + CR;
    if (x2 <> DEF_x2) then Result := Result + Format('  x2(%d)', [x2]) + CR;
    if (y2 <> DEF_y2) then Result := Result + Format('  y2(%d)', [y2]) + CR;
    if (color <> DEF_color) then Result := Result + Format('  color(%d)', [Integer(color)]) + CR;
    if (style <> DEF_style) then Result := Result + Format('  style(%d)', [Integer(style)]) + CR;
end;

// Load position color, and style information from a
// basic serialization.
procedure TDrawingCommand.Deserialize(txt : String);
var
    token_name, token_value : String;
begin
    // Set default values.
    x1 := DEF_x1;
    y1 := DEF_y1;
    x2 := DEF_x2;
    y2 := DEF_y2;
    color := DEF_color;
    style := DEF_style;

    while (txt <> '') do
    begin
        // Read a token from the serialization.
        GetToken(txt, token_name, token_value);

        // See which token it is.
        if (token_name = 'x1') then
            x1 := StrToInt(token_value)
        else if (token_name = 'y1') then
            y1 := StrToInt(token_value)
        else if (token_name = 'x2') then
            x2 := StrToInt(token_value)
        else if (token_name = 'y2') then
            y2 := StrToInt(token_value)
        else if (token_name = 'color') then
            color := TColor(StrToInt(token_value))
        else if (token_name = 'style') then
            style := TBrushStyle(StrToInt(token_value));
    end;
end;

end.
