<?php

global $views;
$views = array(
	'json' => array(
		'class' => 'JSONRenderingEngine',
		'filename' => 'JSON'
	),
	'xhtml' => array(
		'class' => 'XHTMLRenderingEngine',
		'filename' => 'XHTML'
	),
	'xml' => array(
		'class' => 'XMLRenderingEngine',
		'filename' => 'XML'
	)
);

class View {
	public static $METHOD_KEY = 'method';
	public static $METHOD_JSON = 'json';
	public static $METHOD_XHTML = 'xhtml';
	public static $METHOD_XML = 'xml';
	protected static $loaded = array();
	
	/**
	 * An abstraction for pulling the rendering engine key out of the request
	 */
	public static function getMethodFromRequest($request) {
		return (isset($request[self::$METHOD_KEY])) ? $request[self::$METHOD_KEY] : self::$METHOD_XHTML;
	}
	
	/**
	 * Returns an instance of the rendering engine for a given key,
	 * defaulting to XHTML if the requested one does not exist.
	 */
	public static function getRenderingEngine($method) {
		global $views;
		
		if (self::loadRenderingEngine($method)) {
			return new $views[$method]['class']($request);
		} else if (self::loadRenderingEngine(self::$METHOD_XHTML)) {
			return new $views[self::$METHOD_XHTML]['class']($request);
		}
		throw new Exception('Failed to load a rendering engine');
	}
	
	/**
	 * Loads the appropriate rendering engine for the given key
	 */
	protected static function loadRenderingEngine($key) {
		global $views;
		
		// Load already attempted?
		if (isset(self::$loaded[$key])) {
			return self::$loaded[$key];
		}
		
		// Otherwise, check for the file
		if (isset($views[$key])) {
			$path = 'views' . DIRECTORY_SEPARATOR . $views[$key]['filename'] . '.php';
			// And include it
			if (file_exists($path) && is_readable($path) && include $path) {
				// Set a flag as to whether or not this worked rather than
				// redoing all of these steps on the next request for this
				return self::$loaded[$key] = class_exists($views[$key]['class']);
			}
		}
		
		// Failed to find it or reference it, so return false
		return false;
	}
}

class RenderingEngine {
	// The base directory
	protected $context = '.';
	// The name of the templates directory
	protected $templates = 'templates';
	// The filename of the template to display
	protected $template = 'index.php';
	// Give the ability to pass variables explicitly to the template
	protected $variables = array();
	// Override the default value to send headers
	protected $headers = array();
	
	/**
	 * Sets the base directory to include from
	 */
	public function setContext($path) {
		$this->context = $path;
	}
	
	/**
	 * Override the default templates directory
	 */
	public function setTemplatesDirName($dir) {
		$this->templates = $dir;
	}
	
	/**
	 * Used to hand off variables to the template so that it does
	 * not need to know anything about the controller setting the
	 * variable values
	 */
	public function setVariable($key, $value) {
		$this->variables[$key] = $value;
	}
	
	/**
	 * Override the default template name
	 */
	public function setTemplate($filename) {
		$this->template = $filename;
	}
	
	/**
	 * Sends each header/value pair as a complete header
	 */
	public function sendHeaders() {
		foreach ($this->headers as $header => $value) {
			header($header . ': ' . $value);
		}
	}
	
	/**
	 * Changes to the context assigned, so that any include calls
	 * made from within a template will not force the template to
	 * know its own path
	 */
	public function display() {
		// Store the current working directory in a temporary variable
		$cwd = getcwd();
		chdir($this->context);
		$template_path = $this->templates . DIRECTORY_SEPARATOR . $this->template;
		if (file_exists($template_path)) {
			$this->sendHeaders();
			$this->sendHeaders();
			include $template_path;
			chdir($cwd);
		} else {
			chdir($cwd);
			throw new Exception('The template "' . $template_path . '" does not exist.');
		}
	}
}

?>