/**
 *	Copyright (c) 2006 Frozen O Productions
	All rights reserved.
	Written by Shawn Lauriat with much respect for and apologies to Stephen
	"Slug" Russell, Peter Samson, Dan Edwards, and Martin Graetz, Alan Kotok,
	Steve Piner, Robert A Saunders, and Wayne Wiitanen
 
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions are met:
 
	* Redistributions of source code must retain the above copyright notice,
		this list of conditions and the following disclaimer.
	* Redistributions in binary form must reproduce the above copyright notice,
		this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
	* Neither the name of Frozen O Productions nor the names of its
		contributors may be used to endorse or promote products derived from
		this software without specific prior written permission.
 
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
	ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
	LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
	
	Spacewar! was conceived in 1961 by Martin Graetz, Stephen Russell, and
	Wayne Wiitanen. It was first realized on the PDP-1 in 1962 by Stephen
	Russell, Peter Samson, Dan Edwards, and Martin Graetz, together with Alan
	Kotok, Steve Piner, and Robert A Saunders. Spacewar! is in the public
	domain, but this credit paragraph must accompany all distributed versions
	of the program.
	
	This incarnation contains none of the original PDP-1 code, but since it
	(more or less) clones the original game, I felt it best to include the above
	paragraph.
	
	-Shawn
 */

var Constants = {
	"onedegree" : (Math.PI / 180),
	"fullcircle" : (Math.PI / 180) * 360,
	"rates" : {
		"explosion" : 800, // milliseconds it takes to explode
		"position" : 250, // milliseconds between position posting
		"regeneration" : 1000 // milliseconds before a ship regenerates
	}
}

var Draw = {
	arc : function(context, x, y, radius, start, end, clockwise) {
		context.arc(x, y, radius, start, end, clockwise);
	},
	circle : function(context, x, y, radius) {
		context.arc(x, y, radius, 0, (Math.PI / 180) * 360, true);
	},
	/**
	 * points = [
	 * 	   [x1, y1],
	 * 	   [x2, y2],
	 * 	   [x3, y3]
	 * ]
	 */
	polygon : function(context, x, y, points, angle) {
		context.moveTo(
			Math.round(points[0][0] * Math.cos(angle * (Math.PI / 180)) - points[0][1] * Math.sin(angle * (Math.PI / 180)) + x),
			Math.round(points[0][0] * Math.sin(angle * (Math.PI / 180)) + points[0][1] * Math.cos(angle * (Math.PI / 180)) + y)
		);
		for (var p = 1; p < points.length; p++) {
			context.lineTo(
				Math.round(points[p][0] * Math.cos(angle * (Math.PI / 180)) - points[p][1] * Math.sin(angle * (Math.PI / 180)) + x),
				Math.round(points[p][0] * Math.sin(angle * (Math.PI / 180)) + points[p][1] * Math.cos(angle * (Math.PI / 180)) + y)
			);
		}
		context.closePath();
	}
}

function UniverseEvent() { }
UniverseEvent.prototype = new CustomEvent;
UniverseEvent.prototype.type = "universe";

function Universe() {
	var farnsworthParabox = this;
	this.preTick = function() {
		farnsworthParabox.tick();
	}
}
Universe.prototype = new EventDispatcher;
Universe.prototype.events = {
	"init" : [],
	"tick" : []
}
Universe.prototype.height = null;
Universe.prototype.width = null;
Universe.prototype.map = null;
Universe.prototype.frame1 = null;
Universe.prototype.frame2 = null;
Universe.prototype.space1 = null;
Universe.prototype.space2 = null;
Universe.prototype.ticking = false;
Universe.prototype.totalTime = 0;
Universe.prototype.ticks = 0;
Universe.prototype.lastTick = 0;
Universe.prototype.currentTick = 0;
Universe.prototype.tickTime = 0;
Universe.prototype.framerate = null;
Universe.prototype.framerateDisplay = null;
Universe.prototype.objects = [];
Universe.prototype.event = new UniverseEvent; // Reusing the same event throughout

Universe.prototype.toggle = function() {
	if (this.ticking) {
		this.stopTime();
	} else {
		// Attach to start button or something
		this.jumpStartTheSecondBigBang();
	}
}
Universe.prototype.init = function() {
	// Prepare space for interaction
	this.space1 = document.getElementById("space");
	this.space2 = this.space1.cloneNode(true);
	this.frame1 = this.space1.getContext("2d");
	this.frame2 = this.space2.getContext("2d");
	this.space2.style.display = "none";
	this.space1.parentNode.insertBefore(this.space2, this.space1);
	document.body.setAttribute("tabIndex", -1);
	document.body.focus();
	
	// Prepare the map and the contents of the universe
	this.map = this.frame1;
	this.framerateDisplay = document.getElementById("framerate");
	this.event.map = this.map;
	this.height = parseInt(this.space1.getAttribute("height"));
	this.width = parseInt(this.space1.getAttribute("width"));
	this.dispatchEvent("init", this.event);
	this.jumpStartTheSecondBigBang();
}

Universe.prototype.jumpStartTheSecondBigBang = function() {
	this.ticking = true;
	this.tick();
}

Universe.prototype.stopTime = function() {
	this.ticking = false;
}

Universe.prototype.tick = function() {
	if (this.ticking) {
		// Off-display buffer
		this.event.map = this.map = (this.ticks & 2) ? this.frame2 : this.frame1;
		// Draw
		this.draw();
		this.event.framerate = this.framerate;
		this.event.tickTime = this.tickTime;
		this.event.timeStamp = this.currentTick;
		this.dispatchEvent("tick", this.event);
		// Apply the buffer
		if (this.ticks & 2) {
			this.space2.style.display = "block";
			this.space1.style.display = "none";
		} else {
			this.space1.style.display = "block";
			this.space2.style.display = "none";
		}
		// Begin again
		setTimeout(this.preTick, 10);
	}
}

Universe.prototype.draw = function() {
	this.currentTick = new Date().getTime();
	// One second over the difference in time
	this.tickTime = this.currentTick - this.lastTick;
	this.totalTime += 1000 / this.tickTime;
	this.framerate = Math.round(this.totalTime / (++this.ticks));
	// store as fraction of second for strategic math purposes
	this.tickTime /= 1000;
	if (this.lastTick != 0) {
		this.framerateDisplay.firstChild.nodeValue = this.framerate;
	}
	this.map.globalCompositeOperation = "source-over";
	this.map.clearRect(0, 0, this.width, this.height);
	this.lastTick = this.currentTick;
}

var universe = new Universe();
window.addEventListener("load", function() { Universe.prototype.init.apply(universe, arguments) }, false);

/**
 * The wormhole sends spacetime events from one universe to another
 */
function Wormhole() { }
Wormhole.prototype = { };
Wormhole.prototype.me = null;
Wormhole.prototype.you = null;
Wormhole.prototype.dis = null;
Wormhole.prototype.parallelUniverse = null; // Receives messages from another world
Wormhole.prototype.messenger = null; // Sends messages to another world
Wormhole.prototype.data = {};
Wormhole.prototype.lastTick = new Date().getTime();
Wormhole.prototype.tickTime = Wormhole.prototype.lastTick;
Wormhole.prototype.createParallelUniverse = function() {
	var dis = this;
	this.parallelUniverse = request_manager.createAjaxRequest();
	if (this.you == wedge) {
		this.parallelUniverse.get = {"ship" : "wedge"};
	} else {
		this.parallelUniverse.get = {"ship" : "pencil"};
	}
	this.parallelUniverse.addEventListener("data", function() { Wormhole.prototype.updateUniverse.apply(dis, arguments); });
}
Wormhole.prototype.createMessenger = function() {
	var dis = this;
	this.messenger = request_manager.createAjaxRequest();
	if (this.me == wedge) {
		this.messenger.get = {"ship" : "wedge"};
	} else {
		this.messenger.get = {"ship" : "pencil"};
	}
	this.messenger.addEventListener("load", function() { Wormhole.prototype.messageSent.apply(dis, arguments); });
}
Wormhole.prototype.prepare = function(which) {
	if (which == "wedge") {
		this.me = wedge;
		this.you = pencil;
	} else {
		this.me = pencil;
		this.you = wedge;
	}
	var me = this.me;
	document.body.addEventListener("keypress", function() { Ship.prototype.onKeyPress.apply(me, arguments); }, false);
	document.body.addEventListener("keyup", function() { Ship.prototype.onKeyUp.apply(me, arguments); }, false);
	this.open();
	this.sendSnapshot();
}
Wormhole.prototype.open = function() {
	if (!this.parallelUniverse) {
		this.createParallelUniverse();
	}
	this.parallelUniverse.open("GET", "controllers/game/update.php");
	this.parallelUniverse.send();
}
Wormhole.prototype.updateUniverse = function(event) {
	if (event.request.xhr.responseText && event.request.xhr.responseText.length > 2) {
		this.data = eval("("+event.request.xhr.responseText.substring(
			event.request.xhr.responseText.lastIndexOf(
				"\n",
				event.request.xhr.responseText.length - 2
			) + 1
		)+")");
		this.currentTick = new Date().getTime();
		// Total tick = server tick time + request time
		this.tickTime = this.data.tickTime + this.currentTick - this.lastTick;
	}
}
Wormhole.prototype.close = function() {
	request_manager.eliminateAjaxRequest(this.parallelUniverse);
}
Wormhole.prototype.sendSnapshot = function() {
	if (!this.messenger) {
		this.createMessenger();
	}
	this.messenger.open("POST", "controllers/game/move.php");
	this.messenger.post = this.me.position;
	this.messenger.send();
}
Wormhole.prototype.messageSent = function() {
	var dis = this;
	setTimeout(function() { Wormhole.prototype.sendSnapshot.apply(dis); }, 250);
}
// An alias to drawing the affected matter, rather than having two threads clobber each other
Wormhole.prototype.draw = function(event) {
	if (this.lastTick != this.currentTick && this.data.position) {
		this.data.position.rotation = 0;
		this.you.position = this.data.position;
		Ship.prototype.draw.apply(this.you, [event]);
		this.lastTick = this.currentTick;
	}
}
var wormhole = new Wormhole();
universe.addEventListener("tick", function() { Wormhole.prototype.draw.apply(wormhole, arguments) });

function Matter() {
	universe.objects.push(this);
}
Matter.prototype = {};
Matter.prototype.radius = 1;
Matter.prototype.mass = 1;
Matter.prototype.rotationspeed = 90; // degrees per second
Matter.prototype.position = {
	"x" : 0,
	"y" : 0,
	"angle" : 0,
	"xspeed" : 0,
	"yspeed" : 0,
	"acceleration" : 0,
	"rotation" : 0 // 1 = clockwise, -1 = counter, 0 = none
};
// Used for static images
Matter.prototype.displayed = false;
// Only based on radius, make real later
Matter.prototype.hitTest = function(otherMatter) {
	return Math.sqrt(Math.pow(this.position.x - otherMatter.position.x, 2) + Math.pow(this.position.y - otherMatter.position.y, 2)) < this.radius + otherMatter.radius;
}
// Rotate the object clockwise
Matter.prototype.rotateClockwise = function() {
	if (this.position.rotation != 1) {
		this.position.rotation = 1;
	}
}
// Rotate the object counter-clockwise
Matter.prototype.rotateCounterClockwise = function() {
	if (this.position.rotation != -1) {
		this.position.rotation = -1;
	}
}
// Stop rotating the object
Matter.prototype.stopRotation = function() {
	if (this.position.rotation != 0) {
		this.position.rotation = 0;
	}
}
Matter.prototype.accelerate = function() {
	if (this.position.acceleration == 0) {
		this.position.acceleration = 1;
	}
}
Matter.prototype.calculateRotation = function(tickTime) {
	if (this.position.rotation != 0) {
		this.position.angle = (this.position.rotation * this.rotationspeed * tickTime + this.position.angle) % 360;
		if (this.position.angle < 0) {
			this.position.angle += 360;
		}
	}
}
Matter.prototype.calculateSpeed = function(tickTime) {
	// From dead X stop
	if (this.position.xspeed == 0) {
		if (this.position.acceleration != 0) {
			// Start off with 1 to the right or left
			this.position.xspeed = (this.position.angle > 180) ? -1 : 1;
		}
	// Positive X speed
	} else if (this.position.angle < 180) {
		this.position.xspeed += ((90 - Math.abs(this.position.angle - 90)) / 90) * (this.position.acceleration * this.acceleration || 1) * tickTime;
		if (this.position.xspeed > this.topspeed) {
			this.position.xspeed = this.topspeed;
		}
	// Negative X speed
	} else if (this.position.angle > 180) {
		this.position.xspeed -= ((90 - Math.abs(this.position.angle - 270)) / 90) * (this.position.acceleration * this.acceleration || 1) * tickTime;
		if (this.position.xspeed < -this.topspeed) {
			this.position.xspeed = -this.topspeed;
		}
	}
	// From dead Y stop
	if (this.position.yspeed == 0) {
		if (this.position.acceleration != 0 && this.position.angle != 180) {
			// Start off with 1 to up or down
			this.position.yspeed = (this.position.angle - 180 < 0) ? -1 : 1;
		}
	// Positive Y speed
	} else if (this.position.angle != 90 && this.position.angle != 270) {
		this.position.yspeed += ((90 - Math.abs(this.position.angle - 180)) / 90) * (this.position.acceleration * this.acceleration || 1) * tickTime;
		if (this.position.yspeed > this.topspeed) {
			this.position.yspeed = this.topspeed;
		} else if (this.position.yspeed < -this.topspeed) {
			this.position.yspeed = -this.topspeed;
		}
	}
}
Matter.prototype.calculatePosition = function(tickTime) {
	this.position.x += this.position.xspeed * tickTime;
	if (this.position.x > universe.width) {
		this.position.x -= universe.width;
	} else if (this.position.x < 0) {
		this.position.x += universe.width;
	}
	this.position.y += this.position.yspeed * tickTime;
	if (this.position.y > universe.height) {
		this.position.y -= universe.height;
	} else if (this.position.y < 0) {
		this.position.y += universe.height;
	}
}

function Sun() { }
Sun.prototype = new Matter;
Sun.prototype.radius = 10;
Sun.prototype.mass = 10;
Sun.prototype.color = "#ffa500";
Sun.prototype.polygon = [];
Sun.prototype.position = {
	"x" : 0,
	"y" : 0,
	"angle" : 0,
	"xspeed" : 0,
	"yspeed" : 0,
	"acceleration" : 0,
	"rotation" : 1
};
Sun.prototype.draw = function(event) {
	// Sun as the center of the universe
	event.map.save();
	this.calculateRotation(event.tickTime);
	
	event.map.strokeStyle = this.color;
	event.map.beginPath();
	Draw.polygon(event.map, this.position.x, this.position.y, this.polygon, this.position.angle);
	event.map.stroke();
	
	event.map.restore();
}

var sun = new Sun();
sun.polygon = [
	[ -10, 0 ],
	[ -1, -1 ],
	[ 0, -10 ],
	[ 1, -1 ],
	[ 10, 0 ],
	[ 1, 1 ],
	[ 0, 10 ],
	[ -1, 1 ]
];
universe.addEventListener(
	"init",
	function() {
		sun.position.x = universe.width / 2;
		sun.position.y = universe.height / 2;
	}
);
universe.addEventListener("tick", function() { Sun.prototype.draw.apply(sun, arguments) });

function Ship() { }
Ship.prototype = new Matter;
Ship.prototype.radius = 10;
Ship.prototype.mass = 1;
Ship.prototype.color = "#fff";
Ship.prototype.lineWidth = 1;
Ship.prototype.lineCap = "round";
Ship.prototype.polygon = [];
Ship.prototype.topspeed = 50; // pixels per second
Ship.prototype.acceleration = 20;
Ship.prototype.exploded = false;
Ship.prototype.position = {};
Ship.prototype.draw = function(event) {
	if (this.exploded == false) {
		this.drawShip(event);
	} else {
		this.drawExplosion(event);
	}
}
Ship.prototype.drawExplosion = function(event) {
	event.map.save();
	event.map.lineWidth = this.lineWidth;
	event.map.lineCap = this.lineCap;
	event.map.strokeStyle = this.color;
	
	this.percentExploded = (event.timeStamp - this.exploded) / 800;
	
	
	event.map.stroke();
	event.map.restore();
	
}
Ship.prototype.drawShip = function(event) {
	event.map.save();
	event.map.lineWidth = this.lineWidth;
	event.map.lineCap = this.lineCap;
	event.map.strokeStyle = this.color;
	
	this.calculateRotation(event.tickTime);
	this.calculateSpeed(event.tickTime);
	this.calculatePosition(event.tickTime);
	
	event.map.beginPath();
	Draw.polygon(event.map, this.position.x, this.position.y, this.polygon, this.position.angle);
	event.map.stroke();
	event.map.restore();
	
	if (this.blownUp()) {
		this.dieHorribly();
		var dis = this;
		setTimeout(function() { Ship.prototype.regenerate.apply(dis); }, 1000);
	}
}
Ship.prototype.onKeyPress = function(event) {
	switch(event.keyCode) {
		// left
		case 37:
			this.rotateCounterClockwise();
			break;
		// up
		case 38:
			this.accelerate();
			break;
		// right
		case 39:
			this.rotateClockwise();
			break;
	}
}
Ship.prototype.onKeyUp = function(event) {
	switch(event.keyCode) {
		// space
		case 32:
			universe.toggle();
			break;
		// up
		case 38:
			this.position.acceleration = 0;
			break;
		// left
		case 37:
		// right
		case 39:
			this.stopRotation();
			break;
	}
}
Ship.prototype.blownUp = function() {
	// Hit anything?
	for (var i = 0; i < universe.objects.length; i++) {
		if (this.position != universe.objects[i].position && this.hitTest(universe.objects[i])) {
			return true;
		}
	}
	return false;
}
Ship.prototype.dieHorribly = function() {
	this.exploded = new Date().getTime();
}
Ship.prototype.regenerate = function() {
	this.exploded = false;
	this.position = {
		"x" : 20,
		"y" : 20,
		"angle" : 135,
		"xspeed" : 0,
		"yspeed" : 0,
		"acceleration" : 0,
		"rotation" : 0
	};
}

var wedge = new Ship();
// On your mark...
wedge.polygon = [
	[ -3, 7 ],
	[ -5, 10 ],
	[ -5, 4 ],
	[ -3, 2 ],
	[ -2, -5 ],
	[ 0, -10 ],
	[ 2, -5 ],
	[ 3, 2 ],
	[ 5, 4 ],
	[ 5, 10 ],
	[ 3, 7 ]
];
wedge.position = {
	"x" : 20,
	"y" : 20,
	"angle" : 135,
	"xspeed" : 0,
	"yspeed" : 0,
	"acceleration" : 0,
	"rotation" : 0
};

var pencil = new Ship();
// On your mark...
pencil.position.angle = 225;
pencil.polygon = [
	[ -3, 10 ],
	[ -3, 5 ],
	[ -1, 4 ],
	[ -1, -7 ],
	[ 0, -10 ],
	[ 1, -7 ],
	[ 1, 4 ],
	[ 3, 5 ],
	[ 3, 10 ],
	[ 0, 10],
	[ 0, -10 ],
	[ 0, 10 ]
];
pencil.position = {
	"x" : -20,
	"y" : -20,
	"angle" : 315,
	"xspeed" : 0,
	"yspeed" : 0,
	"acceleration" : 0,
	"rotation" : 0
};

universe.addEventListener("tick", function() { Ship.prototype.draw.apply(wedge, arguments) });
universe.addEventListener("tick", function() { Ship.prototype.draw.apply(pencil, arguments) });

/*window.addEventListener("load", function() { document.getElementsById("spacetime").addEventListener("server-time", eventHandler, false); });
function eventHandler(event) {
    document.getElementById("time").firstChild.nodeValue = event.data;
}*/
