<?php

/**
 * Faking namespace in PHP by implementing utility functions as
 * static methods of a Utilities class. This purposefully
 * requires no other custom code, so that the application can
 * load it before anything else.
 */
class Utilities {
	/**
	 * A list of the currently loaded class, since an array
	 * and calls to isset() works MUCH faster than include_once
	 */
	protected static $loaded_classes = array();

	/**
	 * The mapping of what bytes to convert when escaping for markup output.
	 */
	protected static $conv_map = array(0x0,0x2FFFF,0,0xFFFF);
	
	/**
	 * Generate a unique token for the current session,
	 * using a random number and the provided seed.
	 */
	public function generateToken($seed) {
		$random = mt_rand();
		return Utilities::hashWithSalt($random . $seed);
	}
	
	/**
	 * Load an object
	 */
	public static function loadClass($class, $folder = 'objects') {
		// Load already attempted?
		if (isset(self::$loaded_classes[$class])) {
			return self::$loaded_classes[$class];
		}
		
		// Otherwise, check for the file
		if (preg_match('/^[\w_]+$/D', $class) && preg_match('/^[\w_]+$/D', $folder)) {
			$path = $folder . '/' . $class . '.php';
			// And include it
			if (is_readable($path) && include $path) {
				// Set a flag as to whether or not this worked rather than
				// redoing all of these steps on the next request for this
				return self::$loaded_classes[$class] = class_exists($class);
			} else {
				throw new Exception('Failed to load class "' . $class . '" from path "' . $path . '"');
			}
		} else {
			throw new Exception('Class names must only contain word characters or underscores.');
		}
		
		return false;
	}
	
	/**
	 * Load a database object
	 */
	public static function loadModel($class) {
		return self::loadClass($class, 'models');
	}
	
	/**
	 * Escape strings for safe usage in XHTML, XML, etc.
	 */
	public static function escapeXMLEntities($string) {
		return mb_encode_numericentity($string, self::$conv_map, 'UTF-8');
	}
	
	public static function hashWithSalt($string) {
		global $config;
		$salt = $config['settings']['salt'];
		return sha1($string . $salt);
	}
	
	/**
	 * Returns true if the type of $value matches the expected $type,
	 * otherwise throws a DataTypeException
	 */
	public static function assertDataType($type, $value) {
		$matched = false;
		// Since instanceof only works with classes, break out native types
		switch ($type) {
			case 'array':
				$matched = is_array($value);
			case 'bool':
				$matched = is_bool($value);
			case 'float':
				$matched = is_float($value);
			case 'int':
				$matched = is_int($value);
			case 'null':
				$matched = is_null($value);
			case 'numeric':
				$matched = is_numeric($value);
			case 'resource':
				$matched = is_resource($value);
			case 'scalar':
				$matched = is_scalar($value);
			case 'string':
				$matched = is_string($value);
			case 'object':
				$matched = is_object($value);
			// For any other type specified, this will assume a classname
			default:
				$matched = $value instanceof $type;
		}
		if (!$matched) {
			$datatype = (is_object($value) ? get_class($value) : gettype($value));
			throw new DataTypeException(
				'Datatype ' . $type . ' expected, ' . $datatype . ' given.',
				$type,
				$datatype
			);
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true if the size of $value matches the expected $size,
	 * otherwise throws a DataSizeException
	 */
	public static function assertDataSize($size, $value) {
		$matched = false;
		$datasize = 0;
		if (is_string($value)) {
			// This will need to change with PHP6' unicode support
			$matched = ($datasize = strlen($value)) <= $size;
		} else if (is_numeric($value)) {
			$matched = ($datasize = $value) <= $size;
		} else if (!is_scalar($value)) {
			// This method does not currently handle non-scalar values
			throw new InvalidArgumentException(
				'Datatype of string or numeric expected for argument two, ' . gettype($value) . ' given.'
			);
		}
		if (!$matched) {
			throw new DataSizeException(
				'Maximum data size of ' . $size . ' expected, ' . $datasize . ' given.',
				$size,
				$datasize
			);
		} else {
			return true;
		}
	}
	
	/**
	 * Returns a Cache object depending on the type requested,
	 * defaulting to memcache, APC, shmop, and temporary files,
	 * depending on what the system offers.
	 */
	public static function getCache($type = null) {
		global $cache_engines;
		foreach ($cache_engine as $engine => $info) {
			if ((!isset($type) || $type == $engine) && extension_loaded($info['extension'])) {
				return new $info['class']();
			}
		}
		// No matching cache object found
		return false;
	}
	
	/**
	 * Consolidates files into a single file as a cache
	 */
	public static function consolidate($files, $cache) {
		$lastupdated = file_exists($cache) ? filemtime($cache) : 0;
		$generate = false;
		foreach ($files as $file) {
			// Just stop of missing a source file
			if (!file_exists($file)) {
				return false;
			}
			if ($u = filemtime($file) > $lastupdated) {
				$generate = true;
				break;
			}
		}
		// Files changed since the last cache modification
		if ($generate) {
			$temp = tempnam('/tmp', 'cache');
			$temph = fopen($temp, 'w');
			// Now write each of the files to it
			foreach ($files as $file) {
				$fileh = fopen($file);
				while (!feof($fileh)) {
					fwrite($temph, fgets($fileh));
				}
				fclose($fileh);
			}
			fclose($temph);
			rename($temp, $cache);
		}
		return true;
	}
}

global $cache_engines;
$cache_engines = array(
	'memcache' => array(
		'extension' => 'memcache',
		'class' => 'memcacheCache'
	),
	'apc' => array(
		'extension' => 'apc',
		'class' => 'APCCache'
	),
	'shmop' => array(
		'extension' => 'shmop',
		'class' => 'shmopCache'
	),
	'filesystem' => array(
		'extension' => 'standard',
		'class' => 'filesystemCache'
	),
);

/**
 * The simple, abstract class lays out the requirements for objects used
 * by this application to access caching functionality. Each of the methods
 * descibed in this section include a class extending Cache that the
 * application can then use transparently.
 */
abstract class Cache {
	/**
	 * Store the given value in cache, identified by the key and optionally
	 * expiring at a certain time.
	 * @param string $key The identifier for the cached variable
	 * @param mixed $value Any non-resource data to store in cache
	 * @param int $expires An optional timestamp specifying the time at
	 * which the cached value expires. When not given, the value will never
	 * expire.
	 * @return boolean Success
	 */
	abstract public function setCache($key, $value = null, $expires = null);

	/**
	 * Retrieves from cache a previously cached value, transparently taking
	 * the expiration into account as necessary.
	 * @param string $key The identifier for the cached variable
	 * @return mixed|false Previously cached data, or false if the cache
	 * either does not exist or has expired.
	 */
	abstract public function getCache($key);

	/**
	 * Deletes from cache a previously cached value
	 * @param string $key The identifier for the cached variable
	 * @return boolean Returns a boolean as to the success of the deletion.
	 */
	abstract public function deleteCache($key);
}

/**
 * An abstraction class for the memcache extension, which
 * offers much more functionality than exposed here,
 * but this example keeps the object interface generic.
 */
class memcacheCache extends Cache {
	/**
	 * The abstracted memcache instance
	 * @var Memcache $memcache
	 */
	protected $memcache;

	public function setCache($key, $value = null, $expires = null) {
		return $this->memcache->set($key, $value, null, $expires);
	}

	public function getCache($key) {
		return $this->memcache->get($key);
	}

	public function deleteCache($key) {
		return $this->memcache->delete($key);
	}
	
	/**
	 * This simple implementation defaults to one server: localhost.
	 * It could very easily pull in configuration information for
	 * any number of memcache servers.
	 */
	public function __construct() {
		$this->memcache = new Memcache();
		$this->memcache->connect('127.0.0.1', 11211);
	}
}

/**
 * An abstraction class for the APC extension
 */
class APCCache extends Cache {
	public function setCache($key, $value = null, $expires = null) {
		// APC takes a time to live flag rather than a timestamp for expiration
		if (isset($expires)) {
			$expires = $expires - time();
		}
		return apc_store($key, $value, $expires);
	}

	public function getCache($key) {
		return apc_fetch($key);
	}

	public function deleteCache($key) {
		return apc_delete($key);
	}
}

/**
 * An abstraction class for the shmop extension, which
 * implements no serialization or expiration of data,
 * so this class handles it instead.
 */
class shmopCache extends Cache {
	public function setCache($key, $value = null, $expires = null) {
		// If the block already exists, remove it
		$this->deleteCache($key);
		// Create the new block
		$shmop_key = $this->shmopKey($key);
		// Create the serialized value - spikes memory usage for large values
		$shmop_value = serialize($value);
		// Value size + 10 for expiration
		$shmop_size = strlen($shmop_value) + 10;
		// Attempt to open the shmop block, read-only
		if ($block = shmop_open($shmop_key, 'n', 0600, $shmop_size)) {
			$expires = str_pad((int)$expires, 10, '0', STR_PAD_LEFT);
			$written = shmop_write($block, $expires, 0)
					&& shmop_write($block, $shmop_value, 10);
			shmop_close($block);
			return $written;
		}
		return false;
	}

	public function getCache($key) {
		$shmop_key = $this->shmopKey($key);
		// Attempt to open the shmop block, read-only
		if ($block = shmop_open($shmop_key, 'a')) {
			// This object stored the expiration time stamp here
			$expires = (int)shmop_read($block, 0, 10);
			$cache = false;
			// If the expiration time exceeds the current time,
			// return the cache. 
			if (!$expires || $expires > time()) {
				$realsize = shmop_size($block) - 10;
				$cache = unserialize(shmop_read($block, 10, $realsize));
			} else {
				// Close and delete the expired cache
				chmop_delete($block);
			}
			shmop_close($block);
			return $cache;
		}
		return false;
	}

	public function deleteCache($key) {
		$shmop_key = $this->shmopKey($key);
		// Attempt to open the shmop block, read-write
		if ($block = shmop_open($shmop_key, 'w')) {
			$deleted = shmop_delete($block);
			shmop_close($block);
			return $deleted;
		} else {
			// Already gone
			return true;
		}
	}
	
	/**
	 * Keep the key generation all in one place
	 */
	protected function shmopKey($key) {
		return crc32($key);
	}
}

/**
 * An abstraction class for using temporary files for caching
 */
class filesystemCache extends Cache {
	public function setCache($key, $value = null, $expires = null) {
		$filepath = $this->filesystemKey($key);
		// Write the expiration with an exclusive lock to overwrite it
		$expires = str_pad((int)$expires, 10, '0', STR_PAD_LEFT);
		$success = (bool)file_put_contents($filepath, $expires, FILE_EX);
		if ($success) {
			// Append the serialized value to the file
			return (bool)file_put_contents($filepath, serialize($value), FILE_EX | FILE_APPEND);
		}
		return false;
	}

	public function getCache($key) {
		$filepath = $this->filesystemKey($key);
		// Attempt to open the file, read-only
		if (file_exists($filepath) && $file = fopen($filepath, 'r')) {
			// This object stored the expiration time stamp here
			$expires = (int)fread($file, 10);
			// If the expiration time exceeds the current time,
			// return the cache. 
			if (!$expires || $expires > time()) {
				$realsize = filesize($block) - 10;
				$cache = '';
				// Need to read in a loop, since fread returns after 8192 bytes
				while ($chunk = fread($file, $realsize)) {
					$cache .= $chunk;
				}
				fclose($block);
				return unserialize($cache);
			} else {
				// Close and delete the expired cache
				fclose($block);
				$this->deleteCache($key);
			}
		}
		return false;
	}

	public function deleteCache($key) {
		$filepath = $this->filesystemKey($key);
		if (file_exists($filepath)) {
			return unlink($filepath);
		}
		return true;
	}
	
	/**
	 * Keep the key generation all in one place
	 */
	protected function filesystemKey($key) {
		return $this->tempdir . md5($key);
	}
	
	public function __construct() {
		// Could override this to set another directory
		$this->tempdir = sys_get_temp_dir() . md5(__FILE__);
		if (!is_dir($this->tempdir)) {
			mkdir($this->tempdir);
		}
		$this->tempdir .= DIRECTORY_SEPARATOR;
	}
}

class DataSizeException extends Exception {
	public $datasize;
	public $maximum;
	
	public function __construct($message, $datasize, $maximum) {
		$this->datasize = $datasize;
		$this->maximum = $maximum;
		parent::__construct($message);
	}
}

class DataTypeException extends Exception {
	public $datatype;
	public $expected;
	
	public function __construct($message, $datatype, $expected) {
		$this->datatype = $datatype;
		$this->expected = $expected;
		parent::__construct($message);
	}
}

?>
