require "#{File.dirname(__FILE__)}/../test_helper"

class DslUserStoriesTest < ActionController::IntegrationTest
  fixtures :products

  ADAMS_DETAILS = {
    :name       => "Adam Kowalski",
    :address    => "Ul. Główna 123",
    :email      => "klient@skladnica.pl",
    :pay_type   => "check"
  }

  MICHALS_DETAILS = {
      :name     => "Michał Nowak",
      :address  => "Aleja 345",
      :email    => "michal@helion.pl",
      :pay_type => "cc"
  }
  
    
  def setup
    LineItem.delete_all
    Order.delete_all
    @ruby_book = products(:ruby_book)
    @rails_book = products(:rails_book)
  end

  # Użytkownik wchodzi na główną stronę sklepu. Wybiera produkt
  # i dodaje do swojego koszyka. Potem przechodzi do kasy i podaje
  # swoje dane w formularzu zamówienia. Po wysłaniu formularza
  # w bazie danych tworzone jest zamówienie. Zawiera ono dane klienta
  # oraz jedną pozycję, która odpowiada produktowi dodanemu do koszyka.

  def test_buying_a_product
    adam = regular_user
    adam.get "/store/index"
    adam.is_viewing "index"
    adam.buys_a @ruby_book
    adam.has_a_cart_containing @ruby_book
    adam.checks_out ADAMS_DETAILS
    adam.is_viewing "index"
    check_for_order ADAMS_DETAILS, @ruby_book
  end

  def test_two_people_buying
    adam = regular_user
      michal = regular_user
    adam.buys_a @ruby_book
      michal.buys_a @rails_book
    adam.has_a_cart_containing @ruby_book
    adam.checks_out ADAMS_DETAILS
      michal.has_a_cart_containing @rails_book
    check_for_order ADAMS_DETAILS, @ruby_book
      michal.checks_out MICHALS_DETAILS
      check_for_order MICHALS_DETAILS, @rails_book
  end
  

  def regular_user
    open_session do |user|
      def user.is_viewing(page)
        assert_response :success
        assert_template page
      end
      def user.buys_a(product)
        xml_http_request "/store/add_to_cart", :id => product.id
        assert_response :success
      end
      def user.has_a_cart_containing(*products)
        cart = session[:cart]
        assert_equal products.size, cart.items.size
        for item in cart.items
          assert products.include?(item.product)
        end
      end
      def user.checks_out(details)
        post "/store/checkout"
        assert_response :success
        assert_template "checkout"
        post_via_redirect "/store/save_order",
                          :order => {
                            :name     => details[:name],
                            :address  => details[:address],
                            :email    => details[:email],
                            :pay_type => details[:pay_type]
                          }
        assert_response :success
        assert_template "index"
        assert_equal 0, session[:cart].items.size
      end
    end
  end

  def check_for_order(details, *products)
    order = Order.find_by_name(details[:name])
    assert_not_nil order
    
    assert_equal details[:name],     order.name
    assert_equal details[:address],  order.address
    assert_equal details[:email],    order.email
    assert_equal details[:pay_type], order.pay_type
    
    assert_equal products.size, order.line_items.size
    for line_item in order.line_items
      assert products.include?(line_item.product)
    end
  end
end


